package gov.cms.grouper.snf.model.table;

import com.google.common.reflect.ClassPath;
import com.openpojo.reflection.PojoClass;
import com.openpojo.reflection.impl.PojoClassFactory;
import com.openpojo.validation.Validator;
import com.openpojo.validation.ValidatorBuilder;
import com.openpojo.validation.test.impl.GetterTester;
import com.openpojo.validation.test.impl.SetterTester;
import gov.cms.grouper.snf.SnfContext;
import gov.cms.grouper.snf.SnfOption;
import gov.cms.grouper.snf.SnfRuntimeOption;
import gov.cms.grouper.snf.lego.Pair;
import gov.cms.grouper.snf.lego.SnfUtils;
import gov.cms.grouper.snf.lego.Triple;
import gov.cms.grouper.snf.model.Assessment;
import gov.cms.grouper.snf.model.CognitiveLevel;
import gov.cms.grouper.snf.model.PdxEligibility;
import gov.cms.grouper.snf.model.SnfDiagnosisCode;
import gov.cms.grouper.snf.transfer.SnfClaim;
import java.time.format.DateTimeFormatter;
import java.util.Arrays;
import java.util.List;
import java.util.Set;
import java.util.stream.Collectors;
import nl.jqno.equalsverifier.EqualsVerifier;
import nl.jqno.equalsverifier.Warning;
import org.junit.jupiter.api.Assertions;
import org.junit.jupiter.api.Test;

public class ModelBuilderTest {

  @Test
  public void testPdpmLevelFromBimsRowBuilder() {
    CognitiveLevelRow.Builder builder = CognitiveLevelRow.Builder.of();
    String[] data = new String[]{"Moderately Impaired", "100", "", "0", "7"};
    CognitiveLevelRow actual = builder.get(data);
    Assertions.assertEquals(CognitiveLevel.of(data[0]), actual.getCognitiveLevel());
    Assertions.assertEquals(Integer.parseInt(data[1]), actual.getLowVersion().intValue());
    Assertions.assertNull(actual.getHighVersion());
    Assertions.assertEquals(Integer.parseInt(data[3]), actual.getLowScore().intValue());
    Assertions.assertEquals(Integer.parseInt(data[4]), actual.getHighScore().intValue());

    CognitiveLevelRow copy = builder.get(data);
    Assertions.assertTrue(actual.equals(copy));
    Assertions.assertTrue(actual.hashCode() == copy.hashCode());

    data = new String[]{"Moderately Impaired", "100", "", "", ""};
    actual = builder.get(data);

    Assertions.assertEquals(CognitiveLevel.of(data[0]), actual.getCognitiveLevel());
    Assertions.assertEquals(Integer.parseInt(data[1]), actual.getLowVersion().intValue());
    Assertions.assertNull(actual.getHighVersion());
    Assertions.assertNull(actual.getLowScore());
    Assertions.assertNull(actual.getHighScore());

    copy = builder.get(data);
    Assertions.assertTrue(actual.equals(copy));
    Assertions.assertTrue(actual.hashCode() == copy.hashCode());

  }

  @Test
  public void testClinicalCategoryMappingRowBuilder() {
    ClinicalCategoryMappingRow.Builder builder = ClinicalCategoryMappingRow.Builder.of();
    String[] data = new String[]{"Acute Infections", "100", "", "Medical Management",
        "Non-Neurologic"};
    ClinicalCategoryMappingRow row = builder.get(data);
    Assertions.assertEquals(data[0], row.getPdsClinicalCategory());
    Assertions.assertEquals(Integer.parseInt(data[1]), row.getLowVersion().intValue());
    Assertions.assertNull(row.getHighVersion());
    Assertions.assertEquals(data[3], row.getPtOtClinicalCategory());
    Assertions.assertEquals(data[4], row.getSlpClinicalCategory());

    ClinicalCategoryMappingRow copy = builder.get(data);
    Assertions.assertTrue(row.equals(copy));
    Assertions.assertTrue(row.hashCode() == copy.hashCode());

  }

  @Test
  public void testCmgForPtOtRowBuilder() {
    CmgForPtOtRow.Builder builder = CmgForPtOtRow.Builder.of();
    String[] data = new String[]{"test", "100", "", "2", "8", "test2"};
    CmgForPtOtRow row = builder.get(data);
    Assertions.assertEquals(data[0], row.getClinicalCategory());
    Assertions.assertEquals(Integer.parseInt(data[1]), row.getLowVersion().intValue());
    Assertions.assertNull(row.getHighVersion());
    Assertions.assertEquals(Integer.parseInt(data[3]), row.getFunctionScoreLow().intValue());
    Assertions.assertEquals(Integer.parseInt(data[4]), row.getFunctionScoreHigh().intValue());
    Assertions.assertEquals(data[5], row.getCmg());

    CmgForPtOtRow copy = builder.get(data);
    Assertions.assertTrue(row.equals(copy));
    Assertions.assertTrue(row.hashCode() == copy.hashCode());

    builder = CmgForPtOtRow.Builder.of();
    data = new String[]{"testcc", "100", "400", "2", "8", "test156"};
    row = builder.get(data);
    Assertions.assertEquals(data[0], row.getClinicalCategory());
    Assertions.assertEquals(Integer.parseInt(data[1]), row.getLowVersion().intValue());
    Assertions.assertEquals(Integer.parseInt(data[2]), row.getHighVersion().intValue());
    Assertions.assertEquals(Integer.parseInt(data[3]), row.getFunctionScoreLow().intValue());
    Assertions.assertEquals(Integer.parseInt(data[4]), row.getFunctionScoreHigh().intValue());
    Assertions.assertEquals(data[5], row.getCmg());

    Assertions.assertFalse(row.equals(copy));
    Assertions.assertFalse(row.hashCode() == copy.hashCode());

    copy = builder.get(data);
    Assertions.assertTrue(row.equals(copy));
    Assertions.assertTrue(row.hashCode() == copy.hashCode());

  }

  @Test
  public void testPerformanceRecodeRowBuilder() {
    PerformanceRecodeRow.Builder builder = PerformanceRecodeRow.Builder.of();
    String[] data = new String[]{"2", "100", "", "1"};
    PerformanceRecodeRow row = builder.get(data);
    Assertions.assertEquals(Integer.parseInt(data[0]), row.getPerformanceScore().intValue());
    Assertions.assertEquals(Integer.parseInt(data[1]), row.getLowVersion().intValue());
    Assertions.assertNull(row.getHighVersion());
    Assertions.assertEquals(Integer.parseInt(data[3]), row.getFunctionScore().intValue());

    PerformanceRecodeRow copy = builder.get(data);
    Assertions.assertTrue(row.equals(copy));
    Assertions.assertTrue(row.hashCode() == copy.hashCode());

  }

  @Test
  public void testSlpCmgRowBuilder() {
    SlpCmgRow.Builder builder = SlpCmgRow.Builder.of();
    String[] data = new String[]{"0", "100", "", "0", "SA"};
    SlpCmgRow row = builder.get(data);
    Assertions.assertEquals(Integer.parseInt(data[0]), row.getPresenceOfAncSlCi().intValue());
    Assertions.assertEquals(Integer.parseInt(data[1]), row.getLowVersion().intValue());
    Assertions.assertNull(row.getHighVersion());
    Assertions
        .assertEquals(Integer.parseInt(data[3]), row.getMechAltDietOrSwallowDisorder().intValue());
    Assertions.assertEquals(data[4], row.getCmg());

    SlpCmgRow copy = builder.get(data);
    Assertions.assertTrue(row.equals(copy));
    Assertions.assertTrue(row.hashCode() == copy.hashCode());

  }

  @Test
  public void testSlpComorbiditiesRowBuilder() {
    SlpComorbiditiesRow.Builder builder = SlpComorbiditiesRow.Builder.of();
    String[] data = new String[]{"I4900", "100", "", "Hemiplegia or Hemiparesis"};
    SlpComorbiditiesRow row = builder.get(data);
    Assertions.assertEquals(data[0], row.getMdsItem());
    Assertions.assertEquals(Integer.parseInt(data[1]), row.getLowVersion().intValue());
    Assertions.assertNull(row.getHighVersion());
    Assertions.assertEquals(data[3], row.getCategory());

    SlpComorbiditiesRow copy = builder.get(data);
    Assertions.assertTrue(row.equals(copy));
    Assertions.assertTrue(row.hashCode() == copy.hashCode());

  }

  @Test
  public void testDiagnosismasterRowBuilder() {
    DiagnosisMasterRow.Builder builder = DiagnosisMasterRow.Builder.of();
    String[] data = new String[]{"A0102", "100", "", "Acute Infections", "", "Endocarditis", "N/A"};
    DiagnosisMasterRow row = builder.get(data);

    Assertions.assertEquals(data[0], row.getCode());
    Assertions.assertEquals(Integer.parseInt(data[1]), row.getLowVersion().intValue());
    Assertions.assertNull(row.getHighVersion());
    Assertions.assertEquals(data[3], row.getClinicalCategory());
    Assertions.assertNull(row.getSlpCategory());
    Assertions.assertEquals(data[5], row.getNtaCategory());
    Assertions.assertEquals(PdxEligibility.of(data[6]), row.getPdxEligibility());

    DiagnosisMasterRow copy = builder.get(data);
    Assertions.assertTrue(row.equals(copy));
    Assertions.assertTrue(row.hashCode() == copy.hashCode());
  }

  @Test
  public void testNtaCmgRowBuilder() {
    NtaCmgRow.Builder builder = NtaCmgRow.Builder.of();
    String[] data = new String[]{"NE", "100", "", "1", "2"};

    NtaCmgRow row = builder.get(data);

    Assertions.assertEquals(data[0], row.getCmg());
    Assertions.assertEquals(Integer.parseInt(data[1]), row.getLowVersion().intValue());
    Assertions.assertNull(row.getHighVersion());
    Assertions.assertEquals(Integer.parseInt(data[3]), row.getLowScore().intValue());
    Assertions.assertEquals(Integer.parseInt(data[4]), row.getHighScore().intValue());

    NtaCmgRow copy = builder.get(data);
    Assertions.assertTrue(row.equals(copy));
    Assertions.assertTrue(row.hashCode() == copy.hashCode());

  }

  @Test
  public void testNtaComorbidityRowBuilder() {
    NtaComorbidityRow.Builder builder = NtaComorbidityRow.Builder.of();
    String[] data = new String[]{"Bladder and Bowel Appliances: Ostomy", "100", "", "H0100C", "1"};

    NtaComorbidityRow row = builder.get(data);

    Assertions.assertEquals(data[0], row.getConditionService());
    Assertions.assertEquals(Integer.parseInt(data[1]), row.getLowVersion().intValue());
    Assertions.assertNull(row.getHighVersion());
    Assertions.assertEquals(data[3], row.getMdsItems().first());
    Assertions.assertEquals(Integer.parseInt(data[4]), row.getPoint().intValue());

    NtaComorbidityRow copy = builder.get(data);
    Assertions.assertTrue(row.equals(copy));
    Assertions.assertTrue(row.hashCode() == copy.hashCode());

  }

  @Test
  public void testSnfVersionRowBuilder() {
    SnfVersionRow.Builder builder = SnfVersionRow.Builder.of();
    String[] data = new String[]{"104", "20200401", "20200930", "V1.0004"};

    SnfVersionRow row = builder.get(data);
    final DateTimeFormatter formatter = DateTimeFormatter.BASIC_ISO_DATE;

    Assertions.assertEquals(Integer.parseInt(data[0]), row.getInternalVersion().intValue());
    Assertions.assertEquals(SnfUtils.parseDate(formatter, data[1]), row.getFrom());
    Assertions.assertEquals(SnfUtils.parseDate(formatter, data[2]), row.getTo());
    Assertions.assertEquals(data[3], row.getVersion());

    SnfVersionRow copy = builder.get(data);
    Assertions.assertEquals(copy, row);
    Assertions.assertEquals(copy.hashCode(), row.hashCode());
  }

  @Test
  public void testAllTableClass() {
    try {

      ClassLoader loader = this.getClass().getClassLoader();
      ClassPath p = ClassPath.from(loader);
      Set<ClassPath.ClassInfo> cs = SnfUtils
          .toSet(p.getTopLevelClasses(CmgForPtOtRow.class.getPackage().getName()));
      Set<Class<?>> classes = cs.stream().map((info) -> {
        return SnfUtils.doOrDie(() -> Class.forName(info.getName()));
      }).collect(Collectors.toSet());

      List<PojoClass> clazzes = classes.stream().map((cls) -> PojoClassFactory.getPojoClass(cls))
          .collect(Collectors.toList());

      clazzes.add(PojoClassFactory.getPojoClass(Pair.class));
      clazzes.add(PojoClassFactory.getPojoClass(Triple.class));
      clazzes.add(PojoClassFactory.getPojoClass(SnfContext.class));
      clazzes.add(PojoClassFactory.getPojoClass(SnfOption.class));
      clazzes.add(PojoClassFactory.getPojoClass(SnfRuntimeOption.class));
      clazzes.add(PojoClassFactory.getPojoClass(SnfClaim.class));
      clazzes.add(PojoClassFactory.getPojoClass(Assessment.class));
      clazzes.add(PojoClassFactory.getPojoClass(SnfDiagnosisCode.class));
      clazzes.remove(PojoClassFactory.getPojoClass(BasicRow.class));

      Validator validator = ValidatorBuilder.create().with(new SetterTester())
          .with(new GetterTester()).build();
      validator.validate(clazzes);

      // testing equals and hashcode

      EqualsVerifier.simple().forClasses(classes)
          .suppress(Warning.STRICT_INHERITANCE, Warning.INHERITED_DIRECTLY_FROM_OBJECT)
          .suppress(Warning.ALL_FIELDS_SHOULD_BE_USED, Warning.NONFINAL_FIELDS).verify();

      EqualsVerifier.simple().forClasses(Arrays.asList(Pair.class, Triple.class))
          .suppress(Warning.STRICT_INHERITANCE, Warning.INHERITED_DIRECTLY_FROM_OBJECT)
          .suppress(Warning.ALL_FIELDS_SHOULD_BE_USED, Warning.NONFINAL_FIELDS).verify();

    } catch (Throwable ex) {
      ex.printStackTrace();
      Assertions.fail();
    }

  }

}
